"use client";
import { FormEvent, useEffect, useState } from "react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";

export default function ResetPasswordPage() {
  const router = useRouter();
  const { t } = useLanguage();
  const token = router.query.token as string | undefined;

  const [password, setPassword] = useState("");
  const [confirmPassword, setConfirmPassword] = useState("");
  const [err, setErr] = useState<string | null>(null);
  const [loading, setLoading] = useState(false);
  const [verifying, setVerifying] = useState(true);
  const [validToken, setValidToken] = useState(false);
  const [email, setEmail] = useState<string | null>(null);
  const [success, setSuccess] = useState(false);

  useEffect(() => {
    if (!router.isReady) return;
    
    if (!token) {
      setVerifying(false);
      setValidToken(false);
      setErr("No reset token provided");
      return;
    }

    // Verify token
    fetch(`/api/auth/verify-reset-token?token=${token}`)
      .then((r) => r.json())
      .then((j) => {
        setVerifying(false);
        if (j.ok && j.valid) {
          setValidToken(true);
          setEmail(j.email);
        } else {
          setValidToken(false);
          setErr(j.message || "Invalid or expired reset token");
        }
      })
      .catch((e) => {
        setVerifying(false);
        setValidToken(false);
        setErr("Failed to verify reset token");
      });
  }, [token, router.isReady]);

  const onSubmit = async (e: FormEvent) => {
    e.preventDefault();
    setErr(null);

    if (password !== confirmPassword) {
      setErr("Passwords do not match");
      return;
    }

    if (password.length < 6) {
      setErr("Password must be at least 6 characters");
      return;
    }

    if (!token) {
      setErr("No reset token provided");
      return;
    }

    setLoading(true);
    try {
      const res = await fetch("/api/auth/reset-password", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({ token, password }),
      });

      const json = await res.json();

      if (!json.ok) {
        throw new Error(json.error || "Failed to reset password");
      }

      setSuccess(true);
      // Redirect to login after 2 seconds
      setTimeout(() => {
        router.push("/login");
      }, 2000);
    } catch (error: any) {
      setErr(error.message || "Failed to reset password");
    } finally {
      setLoading(false);
    }
  };

  if (verifying) {
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
          <div className='w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center mx-auto mb-4'>
            <svg
              className='w-8 h-8 text-primary animate-spin'
              fill='none'
              viewBox='0 0 24 24'
            >
              <circle
                className='opacity-25'
                cx='12'
                cy='12'
                r='10'
                stroke='currentColor'
                strokeWidth='4'
              ></circle>
              <path
                className='opacity-75'
                fill='currentColor'
                d='M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z'
              ></path>
            </svg>
          </div>
          <h1 className='text-2xl font-bold text-foreground mb-2'>
            Verifying Reset Token
          </h1>
          <p className='text-muted'>Please wait...</p>
        </div>
      </div>
    );
  }

  if (success) {
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
          <div className='w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4'>
            <svg
              className='w-8 h-8 text-green-600'
              fill='none'
              stroke='currentColor'
              viewBox='0 0 24 24'
            >
              <path
                strokeLinecap='round'
                strokeLinejoin='round'
                strokeWidth={2}
                d='M5 13l4 4L19 7'
              />
            </svg>
          </div>
          <h1 className='text-2xl font-bold text-foreground mb-2'>
            Password Reset Successful!
          </h1>
          <p className='text-muted mb-6'>
            Your password has been reset. Redirecting to login...
          </p>
          <Link
            href='/login'
            className='inline-block w-full rounded-xl bg-primary px-4 py-2.5 text-white font-medium hover:opacity-90 transition'
          >
            Go to Login
          </Link>
        </div>
      </div>
    );
  }

  if (!validToken) {
    return (
      <div className='min-h-screen bg-sidebar flex items-center justify-center px-4'>
        <div className='w-full max-w-md rounded-2xl border border-card/60 bg-card shadow-xl p-8 text-center'>
          <div className='w-16 h-16 bg-red-100 rounded-full flex items-center justify-center mx-auto mb-4'>
            <svg
              className='w-8 h-8 text-red-600'
              fill='none'
              stroke='currentColor'
              viewBox='0 0 24 24'
            >
              <path
                strokeLinecap='round'
                strokeLinejoin='round'
                strokeWidth={2}
                d='M6 18L18 6M6 6l12 12'
              />
            </svg>
          </div>
          <h1 className='text-2xl font-bold text-foreground mb-2'>
            Invalid or Expired Token
          </h1>
          <p className='text-muted mb-6'>
            {err || "This password reset link is invalid or has expired."}
          </p>
          <Link
            href='/login'
            className='inline-block w-full rounded-xl bg-primary px-4 py-2.5 text-white font-medium hover:opacity-90 transition'
          >
            Back to Login
          </Link>
        </div>
      </div>
    );
  }

  return (
    <div className='min-h-screen bg-sidebar relative flex items-center justify-center px-4'>
      {/* Background decoration */}
      <div
        aria-hidden
        className='pointer-events-none absolute inset-0 -z-10 overflow-hidden'
      >
        <div className='absolute -top-32 -right-24 w-80 h-80 bg-sidebar-accent/40 rounded-full blur-3xl' />
        <div className='absolute -bottom-32 -left-24 w-80 h-80 bg-primary/20 rounded-full blur-3xl' />
      </div>

      <div className='w-full max-w-md'>
        <div className='rounded-2xl border border-card/60 bg-card shadow-xl'>
          {/* Header */}
          <div className='px-8 pt-8 pb-5 text-center'>
            <div className='mx-auto mb-4 flex h-16 w-16 items-center justify-center rounded-full bg-primary/10'>
              <svg
                className='h-8 w-8 text-primary'
                fill='none'
                stroke='currentColor'
                viewBox='0 0 24 24'
              >
                <path
                  strokeLinecap='round'
                  strokeLinejoin='round'
                  strokeWidth={2}
                  d='M15 7a2 2 0 012 2m4 0a6 6 0 01-7.743 5.743L11 17H9v2H7v2H4a1 1 0 01-1-1v-2.586a1 1 0 01.293-.707l5.964-5.964A6 6 0 1121 9z'
                />
              </svg>
            </div>
            <h1 className='text-3xl font-bold text-foreground mb-1'>
              Reset Password
            </h1>
            {email && (
              <p className='text-muted'>Setting password for {email}</p>
            )}
          </div>

          {/* Form */}
          <form onSubmit={onSubmit} className='px-8 pb-8 space-y-6'>
            {err && (
              <div
                role='alert'
                aria-live='polite'
                className='bg-red-50 border border-red-200 rounded-lg p-4 flex items-start text-sm text-red-800'
              >
                <svg
                  className='w-4 h-4 mr-2 mt-0.5'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z'
                  />
                </svg>
                {err}
              </div>
            )}

            <div className='space-y-4'>
              <div>
                <label
                  htmlFor='password'
                  className='block text-sm font-medium text-foreground mb-2'
                >
                  New Password
                </label>
                <input
                  id='password'
                  type='password'
                  required
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  className='w-full rounded-xl border border-input bg-background px-3.5 py-2.5 text-foreground placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-primary/50'
                  placeholder='Enter new password (min. 6 characters)'
                  autoComplete='new-password'
                  minLength={6}
                />
              </div>

              <div>
                <label
                  htmlFor='confirmPassword'
                  className='block text-sm font-medium text-foreground mb-2'
                >
                  Confirm Password
                </label>
                <input
                  id='confirmPassword'
                  type='password'
                  required
                  value={confirmPassword}
                  onChange={(e) => setConfirmPassword(e.target.value)}
                  className='w-full rounded-xl border border-input bg-background px-3.5 py-2.5 text-foreground placeholder:text-muted focus:outline-none focus:ring-2 focus:ring-primary/50'
                  placeholder='Confirm new password'
                  autoComplete='new-password'
                  minLength={6}
                />
              </div>
            </div>

            <button
              type='submit'
              disabled={loading}
              className='w-full inline-flex items-center justify-center rounded-xl bg-primary px-4 py-2.5 text-white font-medium shadow hover:opacity-90 disabled:opacity-50 disabled:cursor-not-allowed transition'
            >
              {loading ? (
                <>
                  <svg
                    className='animate-spin -ml-1 mr-2 h-5 w-5'
                    xmlns='http://www.w3.org/2000/svg'
                    fill='none'
                    viewBox='0 0 24 24'
                  >
                    <circle
                      className='opacity-25'
                      cx='12'
                      cy='12'
                      r='10'
                      stroke='currentColor'
                      strokeWidth='4'
                    ></circle>
                    <path
                      className='opacity-75'
                      fill='currentColor'
                      d='M4 12a8 8 0 018-8V1C5.373 1 1 5.373 1 12h3z'
                    ></path>
                  </svg>
                  Resetting Password...
                </>
              ) : (
                "Reset Password"
              )}
            </button>
          </form>

          {/* Footer Links */}
          <div className='px-8 pb-6 pt-2 border-t border-card text-center'>
            <p className='text-sm'>
              <Link
                href='/login'
                className='text-muted hover:text-foreground inline-flex items-center justify-center'
              >
                <svg
                  className='w-4 h-4 mr-1'
                  fill='none'
                  stroke='currentColor'
                  viewBox='0 0 24 24'
                >
                  <path
                    strokeLinecap='round'
                    strokeLinejoin='round'
                    strokeWidth={2}
                    d='M10 19l-7-7m0 0l7-7m-7 7h18'
                  />
                </svg>
                Back to Login
              </Link>
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}

